% Example for using time dependent variables in a StateSpaceModel.
% The model used is the same as in example.m, but here time-dependent
% auxiliary states are added. As in example.m, it is recommended to run
% each section using "Run and Advance". Sections from example.m have been
% lumped together

% David Katzin, Wageningen University
% david.katzin@wur.nl

clear;

%% Create a StateSpaceModel object as in example.m
m = StateSpaceModel();

% Define the time span
m.t = DynamicElement('01/01/2001 00:00:00');
m.t.val = [0 48]; % hours

% Define parameters
addParam(m, 'lue', 7.5e-8);
addParam(m, 'heatLoss', 1);
addParam(m, 'heatEff', 0.1);
addParam(m, 'gasPrice', 4.55e-4);
addParam(m, 'lettucePrice', 136.4);
addParam(m, 'heatMin', 0);
addParam(m, 'heatMax', 100);

% Define inputs 
m.d.rad = DynamicElement('d.rad');
m.d.tempOut = DynamicElement('d.tempOut');

% Define states and controls
m.x.dryWeight = DynamicElement('x.dryWeight');
m.x.tempIn = DynamicElement('x.tempIn');
m.u.heating = DynamicElement('u.heating');

% Define auxiliary states
% Photosynthesis [kg m^{-2} h^{-1}], equation 2.1 [1] 
m.a.phot = m.p.lue.*m.d.rad.*m.x.tempIn.*m.x.dryWeight; 

% Heat loss to the outside [degC h^{-1}], equation 2.2 [1] 
m.a.heatOut = m.p.heatLoss.*(m.d.tempOut - m.x.tempIn); 

% Heat gain from the pipes [degC h^{-1}], equation 2.2 [1] 
m.a.heatIn = m.p.heatEff.*m.u.heating;

%% Add time dependent auxiliary states 

% An auxiliary state tracking time. The definition 't' is especially
% reserved for this. Not to be confused with the time variable m.t, which
% saves the timespan of the simulation
m.a.runTime = DynamicElement('t');

% Other ways to use the reserved definition 't'
m.a.doubleTime = DynamicElement('2*t');
m.a.pastTime = DynamicElement('t-10');
m.a.timePhot = DynamicElement('t.*a.phot');

% Time of day (in hours, goes from 0-24, not including 24)
m.a.timeOfDay = mod(m.a.runTime,24);

% 1 if it's currently midnight, 0 otherwise
m.a.isMidnight = ifElse('a.timeOfDay==0',1,0);

% Dry weight at a specific time
m.a.dwAt5 = DynamicElement('atTime(''x.dryWeight'',5)');

% Dry weight one hour ago
m.a.dwHourAgo = DynamicElement('atTime(''x.dryWeight'',t-1)');

% Dry weight 10 hours ago
m.a.dw10HoursAgo = DynamicElement('atTime(''x.dryWeight'',t-10)');

% Dry weight at the start of this day
m.a.dwLastMidnight = DynamicElement('atTime(''x.dryWeight'',t-a.timeOfDay)');

%% Set the ODESs, controls, and inputs

% Photosynthesis [kg m^{-2} h^{-1}], equation 2.1 [1] 
m.x.dryWeight = DynamicElement(m.a.phot); 
% Notice that we used DynamicElement(m.a.phot). 
% This is because simply stating `m.x.dryWeight = m.a.phot` would cause 
% `x.dryWeight` to be equivalent to `a.phot`, and any change to the one would 
% influence the other (recall handle classes). 

% Heat gain in the greenhouse [degC h^{-1}], equation 2.2 [1] 
m.x.tempIn = m.a.heatOut + m.a.heatIn;

% Set controls (as inputs)
time = (0:48)';
m.u.heating.val = [time zeros(size(time))];

% Set the values of the inputs
m.d.rad.val = [time max(0, 800*sin(4*pi*time/48-0.65*pi))];
m.d.tempOut.val = [time 15+10*sin(4*pi*time/48-0.65*pi)];

% Set initial values for the states
m.x.dryWeight.val = 1; % kg m^{-2} 
m.x.tempIn.val = 10; % degC

% Make some plots
% figure; plot(m.d.rad);
% figure; plot(m.d.tempOut);
% figure; plot(m.u.heating);
% plot(m);

%% Simulate
runSim(m,1);
% plot(m);

%% Plot runtime variable
figure;
plot(m.a.runTime);
legend('running time');
xlabel('time (hours)');
ylabel('time (hours)');

%% Other uses of the reserved word 't'
hold on
plot(m.a.doubleTime);
plot(m.a.pastTime);
legend('running time','double time','past time');

%% timePhot
figure;
plot(m.a.timePhot);
hold on
plot(m.a.phot);
legend('timePhot','phot');

%% Plot time dependent variables
figure; hold on;
plot(m.a.runTime);
plot(m.a.timeOfDay);
plot(m.a.isMidnight);
legend('running time','time of day','isMidnight');

%% Plot time dependent dry weight variables
figure; hold on; grid; axis([0 48 0.995 1.025]); xlabel('time (hours)'); ylabel('dry Weight');
plot(m.x.dryWeight);
legend('dryWeight');

%% Dry weight at time point 5 - notice undefined for t<5
plot(m.a.dwAt5);
legend('dryWeight','dryWeight at timepoint 5 (undefined for t<5)');

%% Dry weight one hour ago - notice undefined for t<1
plot(m.a.dwHourAgo);
legend('dryWeight','dryWeight at timepoint 5 (undefined for t<5)',...
    'dryWeight one hour ago (undefined for t<1)');

%% Dry weight 10 hours ago - notice undefined for t<10
plot(m.a.dw10HoursAgo);
legend('dryWeight','dryWeight at timepoint 5 (undefined for t<5)',...
    'dryWeight one hour ago (undefined for t<1)',...
    'dryWeight 10 hours ago (undefined for t<10)');

%% Dry weight at the last midnight
plot(m.a.dwLastMidnight);
legend('dryWeight','dryWeight at timepoint 5 (undefined for t<5)',...
    'dryWeight one hour ago (undefined for t<1)',...
    'dryWeight 10 hours ago (undefined for t<10)',...
    'dryWeight last midnight');